unit uTListImpl;

interface

uses
  Borland.Vcl.SysUtils,
  System.Collections, System.Text;

type
  TListSortCompare = function (Item1, Item2: TObject): Integer;
  TListAssignOp = (laCopy, laAnd, laOr, laXor, laSrcUnique, laDestUnique);
  TList = class(TObject)
  private
    FList: System.Collections.ArrayList;
  protected
    function Get(Index: Integer): TObject;
    function GetCount: Integer;
    function GetCapacity: Integer;
    procedure Grow; virtual;
    procedure Put(Index: Integer; Item: TObject);
    procedure SetCapacity(NewCapacity: Integer);
    procedure SetCount(NewCount: Integer);
  public
    constructor Create;
    function Add(Item: TObject): Integer;
    procedure Clear; virtual;
    procedure Delete(Index: Integer);
    // TODO: Class overloaded virtuals
    class procedure Error(const Msg: string; Data: Integer); overload; //virtual;
    procedure Exchange(Index1, Index2: Integer);
    function Expand: TList;
    function Extract(Item: TObject): TObject;
    function First: TObject;
    function IndexOf(Item: TObject): Integer;
    procedure Insert(Index: Integer; Item: TObject);
    function Last: TObject;
    procedure Move(CurIndex, NewIndex: Integer);
    function Remove(Item: TObject): Integer;
    procedure Pack;
    procedure Sort(Compare: TListSortCompare);
    procedure Assign(ListA: TList; AOperator: TListAssignOp = laCopy; ListB: TList = nil);
    property Capacity: Integer read GetCapacity write SetCapacity;
    property Count: Integer read GetCount write SetCount;
    property Items[Index: Integer]: TObject read Get write Put; default;
    property List: System.Collections.ArrayList read FList;
  end;


implementation

resourcestring
  SListCapacityError = 'List capacity out of bounds (%d)';
  SListCountError = 'List count out of bounds (%d)';
  SListIndexError = 'List index out of bounds (%d)';

{ TListComparer }

type
  EListError = class(Exception);
  TListComparer = class(TObject, IComparer)
  private
    FCompare: TListSortCompare;
  public
    function Compare(O1, O2: TObject): Integer;
    constructor Create(Compare: TListSortCompare);
  end;

function TListComparer.Compare(O1, O2: TObject): Integer;
begin
  Result := FCompare(O1, O2);
end;

constructor TListComparer.Create(Compare: TListSortCompare);
begin
  inherited Create;
  FCompare := Compare;
end;

{ TList }

constructor TList.Create;
begin
  inherited Create;
  FList := System.Collections.ArrayList.Create;
end;

function TList.Add(Item: TObject): Integer;
begin
  Result := FList.Add(Item);
end;

procedure TList.Clear;
begin
  FList.Clear;
end;

procedure TList.Delete(Index: Integer);
var
  Temp: TObject;
begin
  Temp := FList[Index];
  FList.RemoveAt(Index);
end;

class procedure TList.Error(const Msg: string; Data: Integer);
begin
  raise EListError.CreateFmt(Msg, [Data]);
end;

{
class procedure TList.Error(Msg: PResStringRec; Data: Integer);
begin
  TList.Error(LoadResString(Msg), Data);
end;
}

procedure TList.Exchange(Index1, Index2: Integer);
var
  Item: TObject;
begin
  Item := FList[Index1];
  FList[Index1] := FList[Index2];
  FList[Index2] := Item;
end;

function TList.Expand: TList;
begin
  if FList.Count = FList.Capacity then
    Grow;
  Result := Self;
end;

function TList.First: TObject;
begin
  Result := Get(0);
end;

function TList.Get(Index: Integer): TObject;
begin
  Result := FList[Index];
end;

function TList.GetCapacity: Integer;
begin
  Result := FList.Capacity;
end;

function TList.GetCount: Integer;
begin
  Result := FList.Count;
end;

procedure TList.Grow;
var
  Delta: Integer;
  LCapacity: Integer;
begin
  LCapacity := FList.Capacity;
  if LCapacity > 64 then
    Delta := LCapacity div 4
  else
    if LCapacity > 8 then
      Delta := 16
    else
      Delta := 4;
  SetCapacity(LCapacity + Delta);
end;

function TList.IndexOf(Item: TObject): Integer;
begin
  Result := FList.IndexOf(TObject(Item));
end;

procedure TList.Insert(Index: Integer; Item: TObject);
begin
  FList.Insert(Index, Item);
end;

function TList.Last: TObject;
begin
  Result := Get(Count - 1);
end;

procedure TList.Move(CurIndex, NewIndex: Integer);
var
  Item: TObject;
begin
  if CurIndex <> NewIndex then
  begin
    if (NewIndex < 0) or (NewIndex >= Count) then
      Error(SListIndexError, NewIndex);
    Item := Get(CurIndex);
    FList.RemoveAt(CurIndex);
    FList.Insert(NewIndex, Item);
  end;
end;

procedure TList.Put(Index: Integer; Item: TObject);
var
  Temp: TObject;
begin
  if (Index < 0) or (Index >= Count) then
    Error(SListIndexError, Index);
  if Item <> FList[Index] then
  begin
    Temp := FList[Index];
    FList[Index] := Item;
  end;
end;

function TList.Remove(Item: TObject): Integer;
begin
  Result := IndexOf(Item);
  if Result >= 0 then
    Delete(Result);
end;

procedure TList.Pack;
var
  I: Integer;
begin
  for I := Count - 1 downto 0 do
    if Items[I] = nil then
      Delete(I);
end;

procedure TList.SetCapacity(NewCapacity: Integer);
begin
  if NewCapacity < Count then
    Error(SListCapacityError, NewCapacity);
  FList.Capacity := NewCapacity;
end;

procedure TList.SetCount(NewCount: Integer);
var
  C: Integer;
  TempArray: array of System.Object;
begin
  if NewCount < 0 then
    Error(SListCountError, NewCount);
  C := FList.Count;
  if NewCount > C then
  begin
    SetLength(TempArray, NewCount - C);
    FList.AddRange(System.Object(TempArray) as ICollection);
  end
  else
  begin
    SetLength(TempArray, C - NewCount);
    FList.CopyTo(TempArray, NewCount);
    FList.RemoveRange(NewCount, C - NewCount);
  end;
end;

procedure TList.Sort(Compare: TListSortCompare);
begin
  FList.Sort(TListComparer.Create(Compare));
end;

function TList.Extract(Item: TObject): TObject;
var
  I: Integer;
begin
  Result := nil;
  I := IndexOf(Item);
  if I >= 0 then
  begin
    Result := Item;
    FList.RemoveAt(I);
  end;
end;

procedure TList.Assign(ListA: TList; AOperator: TListAssignOp; ListB: TList);
var
  I: Integer;
  LTemp, LSource: TList;
begin
  // ListB given?
  if ListB <> nil then
  begin
    LSource := ListB;
    Assign(ListA);
  end
  else
    LSource := ListA;

  // on with the show
  case AOperator of

    // 12345, 346 = 346 : only those in the new list
    laCopy:
      begin
        Clear;
        Capacity := LSource.Capacity;
        for I := 0 to LSource.Count - 1 do
          Add(LSource[I]);
      end;

    // 12345, 346 = 34 : intersection of the two lists
    laAnd:
      for I := Count - 1 downto 0 do
        if LSource.IndexOf(Items[I]) = -1 then
          Delete(I);

    // 12345, 346 = 123456 : union of the two lists
    laOr:
      for I := 0 to LSource.Count - 1 do
        if IndexOf(LSource[I]) = -1 then
          Add(LSource[I]);

    // 12345, 346 = 1256 : only those not in both lists
    laXor:
      begin
        LTemp := TList.Create; // Temp holder of 4 byte values
        LTemp.Capacity := LSource.Count;
        for I := 0 to LSource.Count - 1 do
          if IndexOf(LSource[I]) = -1 then
            LTemp.Add(LSource[I]);
        for I := Count - 1 downto 0 do
          if LSource.IndexOf(Items[I]) <> -1 then
            Delete(I);
        I := Count + LTemp.Count;
        if Capacity < I then
          Capacity := I;
        for I := 0 to LTemp.Count - 1 do
          Add(LTemp[I]);
      end;

    // 12345, 346 = 125 : only those unique to source
    laSrcUnique:
      for I := Count - 1 downto 0 do
        if LSource.IndexOf(Items[I]) <> -1 then
          Delete(I);

    // 12345, 346 = 6 : only those unique to dest
    laDestUnique:
      begin
        LTemp := TList.Create;
        LTemp.Capacity := LSource.Count;
        for I := LSource.Count - 1 downto 0 do
          if IndexOf(LSource[I]) = -1 then
            LTemp.Add(LSource[I]);
        Assign(LTemp);
      end;
  end;
end;

end.


